/****************************************************************************
*
*    Copyright (c) 2005 - 2014 by Vivante Corp.  All rights reserved.
*
*    The material in this file is confidential and contains trade secrets
*    of Vivante Corporation. This is proprietary information owned by
*    Vivante Corporation. No part of this work may be disclosed,
*    reproduced, copied, transmitted, or used in any way for any purpose,
*    without the express written permission of Vivante Corporation.
*
*****************************************************************************/


/*!
 @header tinyUI Utilities
 Some common utilities that can be used by programs to ease programming with the tinyUI API.

 @copyright Vivante Corporation
 */

#ifndef _tiny_ui_util_h_
#define _tiny_ui_util_h_

#include "tiny_ui.h"

#ifdef __cplusplus
extern "C" {
#endif

/*!
 @abstract Load a PNG file into a buffer.

 @discussion
 Using the PNG library, a PNG file will be read into memory and a <code>tiny_ui_buffer_t</code> structure will be filled in with
 the information about this PNG file.

 Normal kernel allocation will happen for the PNG file bits, so the buffer can later be freed by the {@link tiny_ui_free}
 function when it is no longer required.

 @param buffer
 A pointer to a <code>tiny_ui_buffer_t</code> structure that will be filled in on success.

 @param name
 The name of the PNG file.

 @result
 A boolean value indicating success (1) or failure (0).
 */
int tiny_ui_load_png(tiny_ui_buffer_t * buffer, const char * name);

/*!
 @abstract Save a buffer to a PNG file.

 @discussion
 Using the PNG library, the specified buffer will be written into a PNG.

 @param name
 The name of the PNG file.

 @param buffer
 A pointer to a <code>tiny_ui_buffer_t</code> structure that holds the image that needs to written.

 @result
 A boolean value indicating success (1) or failure (0).
 */
int tiny_ui_save_png(const char * name, tiny_ui_buffer_t * buffer);

/*!
 @abstract Load an identity matrix.

 @discussion
 Load an identity matrix into a matrix variable.

 @param matrix
 Pointer to a <code>tiny_ui_matrix_t</code> structure that will be loaded with an identity matrix.
 */
void tiny_ui_identity(tiny_ui_matrix_t * matrix);

/*!
 @abstract Translate a matrix.

 @discussion
 Translate a matrix to a new position.

 @param x
 X location of the transformation.

 @param y
 Y location of the transformation.

 @param matrix
 Pointer to a <code>tiny_ui_matrix_t</code> structure that will be translated.
 */
void tiny_ui_translate(tiny_ui_float_t x, tiny_ui_float_t y, tiny_ui_matrix_t * matrix);

/*!
 @abstract Scale a matrix.

 @discussion
 Scale a matrix in both x and y directions.

 @param scale_x
 Horizontal scale.

 @param scale_y
 Vertical scale.

 @param matrix
 Pointer to a <code>tiny_ui_matrix_t</code> structure that will be scaled.
 */
void tiny_ui_scale(tiny_ui_float_t scale_x, tiny_ui_float_t scale_y, tiny_ui_matrix_t * matrix);

/*!
 @abstract Rotate a matrix.

 @discussion
 Rotate a matrix a certain number of degrees.

 @param degrees
 Number of degrees to rotate the matrix around. Positive numbers rotate counter clock wise.

 @param matrix
 Pointer to a <code>tiny_ui_matrix_t</code> structure that will be rotated.
 */
void tiny_ui_rotate(tiny_ui_float_t degrees, tiny_ui_matrix_t * matrix);

/*!
 @abstract Get a buffer pointing to the frame buffer if any.

 @discussion
 On most devices their will be a frame buffer attachment. This function wraps the complexities of using the frame buffer device
 as an output. The buffer returned holds the information about the frame buffer. When the frame buffer is no longer needed,
 {@link tiny_ui_fb_close} should be called with the buffer returned by this function.

 If the device doesn't support a frame buffer, <code>NULL</code> will be returned.

 @result
 Returns a pointer to a <code>tiny_ui_buffer_t</code> structure which can be used as a normal buffer on success. On failure
 <code>NULL</code> will be returned.
 */
tiny_ui_buffer_t * tiny_ui_fb_open(void);

/*!
 @abstract Get a buffer pointing to the frame buffer if any.

 @discussion
 On most devices their will be a frame buffer attachment. This function wraps the complexities of using the frame buffer device
 as an output. The buffer returned holds the information about the frame buffer. When the frame buffer is no longer needed,
 {@link tiny_ui_fb_close} should be called with the buffer returned by this function.

 If the device doesn't support a frame buffer, <code>NULL</code> will be returned.

 @result
 Returns a pointer to a <code>tiny_ui_buffer_t</code> structure which can be used as a normal buffer on success. On failure
 <code>NULL</code> will be returned.
 */
void tiny_ui_fb_close(tiny_ui_buffer_t * buffer);

/*!
 @abstract Upload a path to GPU memory.

 @discussion
 In normal cases, the tinyUI driver will copy any path data into a command buffer structure during runtime. This does take some
 time if there are many paths to be rendered. Also, in an embedded system the path data wont change - so it makes sense to
 upload the path data into GPU memory in such a form the GPU can directly access it.

 This function will allocate a buffer that will contain the path data and the required command buffer header and footer data for
 the GPU to access the data directly.

 @param path
 Pointer to a <code>tiny_ui_path_t</code> structure that contains the path to be uploaded. Some fields in this structure will be
 modified to point to a command buffer instead of the native path data.

 @result
 A pointer to a <code>tiny_ui_buffer_t</code> structure that contains the command buffer and path data after uploading it to GPU
 memory. <code>NULL</code> is returned if there is an error.
 */
tiny_ui_buffer_t * tiny_ui_upload_path(tiny_ui_path_t * path);

#ifdef __cplusplus
}
#endif
#endif /* _tiny_ui_util_h_ */
